require 'csv'

# Uma tabela =]
class Table
	# Cabecalho da tabela
	attr_reader :header
	# Hash com valores que cada atributo pode ter, ex.: { "attr_name" => [valor1, valor2], ...}
	attr_reader :values
	# Contedo da tabela
	attr_reader :data
	# Entropia da tabela
	attr_reader :entropy

	# Constructor, recebe o cabealho da tabela e os dados, opcionalmente pode
	# receber os possiveis valores de cada atributo, caso no seja dado a tabela
	# ser varrida e eles sero calculados
	def initialize(header, data, values = nil)
		@header = header
		@data = data

		if values.nil?
			# Verifica os possiveis valores de cada atributo
			# attr_name => value_set
			@values = {}
			@header.each_with_index do |attr, attr_idx|
				@values[attr] = Set.new
				@data.each do |row|
					@values[attr] << row[attr_idx]
				end
			end
		else
			@values = values
		end
		# calcula entropia
		@entropy = calc_entropy
	end

	# Cria uma tabela a partir de um arquivo CSV
	def self.fromCSV(csv_file)
		data = CSV.open(csv_file, 'r').entries
		raise "Dados insuficientes" if data.length < 2
		header = data.shift
		return Table.new(header, data)
	end

	def num_rows
		@data.length
	end

	def empty?
		@data.empty?
	end

	def value_of(row, col)
		@data[row][col]
	end

	# calcula o ganho sobre o atributo com id attr_idx
	def gain(attr_idx)
		attr_name = @header[attr_idx]

		gain = entropy
		@values[attr_name].each do |attr_value|
			sub_table = subset(attr_idx, attr_value)
			gain -= (sub_table.num_rows/num_rows.to_f)*sub_table.entropy
		end
		return gain
	end

	# Retorna uma tabela com o sub conjunto dos dados onde todo mundo tem o valor value no atributo attr
	def subset(attr_idx, attr_value)
		sub_data = @data.select do |row|
			row[attr_idx] == attr_value
		end
		return Table.new(@header, sub_data, @values)
	end

private
	# log na base 2
	def log2(val)
		return val == 0 ? 0 : Math.log(val)/Math.log(2)
	end

	# calcula a entropia da tabela
	def calc_entropy
		n = @header.length - 1
		results = {}
		@data.each do |row|
			if results.include? row[n]
				results[row[n]] += 1
			else
				results[row[n]] = 1
			end
		end
		entropy = 0.0;
		results.each do |k,v|
			prob = v/@data.length.to_f
			entropy -= prob*log2(prob)
		end
		return entropy
	end
end
