#!/usr/bin/env ruby

require 'csv'

def show_help
	puts """Para gerar o grafo da rvore de deciso use:
  #{$0} creategraph CSV_FILE PNG_OUTPUT

Para imprimir a rvore de deciso na tela use:
  #{$0} dumptree CSV_FILE

Para executar a rvore de deciso use:
  #{$0} execute CSV_FILE

Para testar a rvore de deciso contra uma base de dados use:
  #{$0} test CSV_FILE CSV_TEST_FILE

Onde:
  CSV_FILE  um arquivo no formato CSV que ser usado para criar a rvore de deciso.
  PNG_OUTPUT  o nome do arquivo PNG que ser gerado aps a execuo do programa.
  CSV_TEST_FILE  um arquivo no formato CSV que ser usado como base de testes para a rvore de deciso
"""
	exit 1
end

def create_graph(csv_file, png_file)
	tree = DecisionTree.new(csv_file)
	proc = open("|dot -Tpng -o#{png_file}", "wb")
	proc.write(tree.dot_script)
	proc.close
end

def show_tree(csv_file)
	DecisionTree.new(csv_file).root.dump_tree
end

def execute(csv_file)
	tree = DecisionTree.new(csv_file)

	puts "Pergunta que no quer calar: \"#{tree.question}?\""

	node = tree.root
	answer = "No sei!!!!"

	while not node.nil?
		if node.children.empty?
			answer = node.name
			break
		end
		puts "#{node.name}? [#{node.children.keys.join('|')}]"
		user_answer = $stdin.gets
		node = node.children[user_answer.strip]
	end
	puts "Resposta: #{answer}"
end

def test_tree(csv_data_file, csv_test_file)
	tree = DecisionTree.new(csv_data_file)
	ok, total = tree.test_against(csv_test_file)
	
	puts """Nmero de testes: #{total}
Acertos: #{ok*100.0/total}% (#{ok})
Erros: #{(total-ok)*100.0/total}% (#{total-ok})
"""
end

if $0 == __FILE__
	Dir.chdir(File.dirname(__FILE__))
	require 'decisiontree'

	show_help if ARGV.length < 2

	begin
		case ARGV[0]
			when "creategraph"
				show_help if ARGV.length < 3
				create_graph(ARGV[1], ARGV[2])
			when "dumptree"
				show_tree(ARGV[1])
			when "execute"
				execute(ARGV[1])
			when "test"
				show_help if ARGV.length < 3
				test_tree(ARGV[1], ARGV[2])
			else
				show_help
		end
	rescue
		puts "Erro: #{$!}"
	end
end
