require 'set'
require 'table'

# Um n da rvore de deciso
class DecisionNode
	attr_reader :name
	attr_reader :id
	attr_reader :children

	@@nextid = 0

	def initialize(name)
		@name = name
		@children = {}
		@id = @@nextid
		@@nextid += 1
	end

	# Adiciona um filho  rvore
	# arc_name - o nome do arco
	# node - o n filho
	def append_child(arc_name, node)
		@children[arc_name] = node
	end

	def children?
		not @children.empty?
	end

	# Imprime a rvore em stdout
	def dump_tree
		print_tree(self, 0)
	end

	# Retorna um script (dot) descrevendo o grafo formado por este n e seus filhos.
	def dot_script
		buffer = String.new
		buffer << "digraph G {\n";
		buffer << "node [shape=box, style=rounded, fontsize=12, height=0.2]\n";
		buffer << "edge [fontsize=12]\n"
		generate_dot_stript(self, buffer)
		buffer << "}\n"
		return buffer
	end

private
	def generate_dot_stript(node, buffer)
		style = node.children.empty? ? ', color=red' : ''
		buffer << "node#{node.id} [label=\"#{node.name}\" #{style}]\n"
		node.children.each do |arc_name, child|
			generate_dot_stript(child, buffer)
			buffer << "node#{node.id} -> node#{child.id} [label=#{arc_name}];\n"
		end
	end

	def print_tree(node, deep)
		puts node.name
		node.children.each do |arc_name, child|
			(deep*10).times{ putc(' ') }
			print arc_name, '=>'
			print_tree(child, deep+1)
		end
	end
end

# rvore de deciso
class DecisionTree
	# Raiz da rvore
	attr_reader :root
	attr_reader :question

	# ctor, receve o nome de um arquivo no formato CSV
	def initialize(csv_file)
		table = Table.fromCSV(csv_file)

		@question = table.header.last
		attrs = Set.new((0..(table.header.length-2)).to_a);
		raise "Dados viciados" if table.entropy.zero?
		expand_node(nil, nil, table, attrs)
	end

	def dot_script
		@root.dot_script
	end

	def test_against(csv_file)
		resultados_ok = 0

		table = Table.fromCSV(csv_file)
		table.data.each do |row|
			node = @root
			while node.children?
				idx = table.header.rindex(node.name)
				node = idx.nil? ? nil : node.children[row[idx]]
				break if node.nil?
			end
			resultados_ok += 1 if not node.nil? and node.name == row[table.header.length - 1]
		end
		return resultados_ok, table.data.length
	end

private
	# expande os ns da rvore at a morte
	def expand_node(arc_name, parent, table, attrs)
		return if table.empty?

		# Se a entropia da tabela  zero  por que temos um n folha
		if table.entropy.zero?
			value = table.value_of(0, table.header.length-1)
			node = DecisionNode.new(value)
			parent.append_child(arc_name, node)
			return
		end

		# calcula todos os ganhos para saber quem  o prx. n
		gain = {}
		attrs.each do |attr_idx|
			gain[attr_idx] = table.gain(attr_idx)
		end
		# Ordena os ganhos para saber qual o melhor
 		best_attr_idx, best_gain = gain.sort {|x, y| x[1] <=> y[1]}.pop
		best_attr_name = table.header[best_attr_idx]
		# depois que escolheu tira ele da tabela
		attrs.delete(best_attr_idx)
		# cria o n da rvore
		node = DecisionNode.new(best_attr_name)
		if parent.nil?
			@root = node
		else
			parent.append_child(arc_name, node)
		end

		# Tenta criar mais ns filhos
		table.values[best_attr_name].each do |value|
				subtable = table.subset(best_attr_idx, value)
				expand_node(value, node, subtable, attrs)
		end
 		attrs << best_attr_idx
	end
end
