package sudoku.jogo;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * Tabuleiro do jogo Sudoku.
 * 
 * @author Ccero Alan Leite Cruz, alan at lsd.ufcg.edu.br, 27/09/2007
 *
 */
public class Tabuleiro {
	
	private Celula[][] array;
	private int tamanho;
	private int xAnterior;
	private int yAnterior;
	private List<ITabuleiroListener> listeners;
	
	/**
	 * @param tamanho
	 */
	public Tabuleiro(int tamanho){
		this.tamanho = tamanho;
		this.xAnterior = 0;
		this.yAnterior = 0;
		array = new Celula[tamanho][tamanho];
		this.listeners = new ArrayList<ITabuleiroListener>();
		inicializaArray();
		jogoUm();
	}

	private void inicializaArray() {
		for(int i = 0 ; i < tamanho ; i ++){
			for(int j = 0 ; j < tamanho; j ++){
				array[i][j] = new Celula(i,j,tamanho);
			}
		}
	}
	
	/**
	 * Atribui um valor a uma clulas respeitando as suas restries.
	 * 
	 * @param x
	 * @param y
	 * @param valor
	 * @return
	 */
	public boolean setCelula(int x, int y, int valor){
		if(naoRestricoes(x,y,valor,true)){
			if(!array[x][y].setValor(new Integer(valor))) return false;
			propagaRestricoes(x,y,valor);
			this.xAnterior = x;
			this.yAnterior = y;
			notifyListeners();
			return true;
		}
		return false;
	}
	
	/**
	 * Atribui um novo valor a uma clula que j foi atribuda anteriormente.
	 * 
	 * @param x
	 * @param y
	 * @param valor
	 * @return
	 */
	public boolean setValor(int x, int y, int valor){
		if(array[x][y].isFixo()){
			return false;
		}
		if(array[x][y].getValor() != 0) {
			retornaRestricoes(x, y, array[x][y].getValor());
		}
		return setCelula(x, y, valor);
	}
	
	/**
	 * @param x
	 * @param y
	 * @param valor
	 */
	private void propagaRestricoes(int x, int y, int valor) {
		propagaRestricaoLinha(x,y,valor);
		propagaRestricaoColuna(x,y,valor);
		propagaRestricaoQuadrante(x,y,valor);
	}

	private void propagaRestricaoQuadrante(int x, int y, int valor) {
		switch (getQuadrante(x,y)) {
		case 1:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 2:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = tamanho/3 ; j < 2*tamanho/3 ; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor));
				}
			}
			break;
		case 3:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor));
				}
			}
			break;
		case 4:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 5:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 6:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 7:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 8:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		case 9:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y) array[i][j].removeValor(new Integer(valor)); 
				}
			}
			break;
		}	
	}

	private void propagaRestricaoLinha(int x, int y, int valor) {
		for(int i = 0 ; i < tamanho ; i++){
			if(i != y) array[x][i].removeValor(new Integer(valor)); 
		}
	}

	private void propagaRestricaoColuna(int x, int y, int valor) {
		for(int i = 0 ; i < tamanho ; i++){
			if(i != x) array[i][y].removeValor(new Integer(valor)); 
		}
	}

	/**
	 * Verifica se no h restries para um valor em uma determinada clula.
	 * 
	 * @param x
	 * @param y
	 * @param valor
	 * @param controle
	 * @return
	 */
	private boolean naoRestricoes(int x,int y,int valor, boolean controle){
		boolean linha = restricaoLinha(x,y,valor,controle);
		boolean coluna = restricaoColuna(x,y,valor,controle);
		boolean quadrante = restricaoQuadrante(x,y,valor,controle);
		return linha && coluna && quadrante;
	}

	private boolean restricaoQuadrante(int x, int y, int valor,boolean controle) {

		switch (getQuadrante(x,y)) {
		case 1:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle && !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
				}
			}
			return true;
		case 2:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = tamanho/3 ; j < 2*tamanho/3 ; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle && !cel.isVerificado() &&  cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
				}
			}
			return true;
		case 3:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
				}
			}
			return true;
		case 4:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		case 5:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		case 6:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		case 7:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		case 8:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		case 9:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.isVerificado() && cel.getValor() == valor && (i != x && j != y)) return false; 
					if(controle &&  !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && (i != x && j != y)) return false;
			
				}
			}
			return true;
		}
		return true;
	}

	public int getQuadrante(int x, int y) {
		
		if(x < tamanho/3){
			if(y < tamanho/3) return 3;
			else if(y >= 2*tamanho/3) return 1;
			else return 2;
		}else if(x >= 2*tamanho/3){
			if(y < tamanho/3) return 9;
			else if(y >= 2*tamanho/3) return 7;
			else return 8;
		}else{
			if(y < tamanho/3) return 4;
			else if(y >= 2*tamanho/3) return 6;
			else return 5;
		}
	}

	private boolean restricaoColuna(int x, int y, int valor, boolean controle) {
		for(int i = 0 ; i < tamanho ; i++){
			Celula cel = this.array[i][y];
			if(cel.isVerificado() && cel.getValor().intValue() == valor && i != x) return false; 
			if(controle && !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && i != x) return false;
		}
		return true;
	}

	private boolean restricaoLinha(int x, int y, int valor, boolean controle) {
		for(int i = 0 ; i < tamanho ; i++){
			Celula cel = this.array[x][i];
			if(cel.isVerificado() && i != y && cel.getValor().intValue() == valor) return false; 
			if(controle && !cel.isVerificado() && cel.pertence(new Integer(valor))>=0 && cel.getCusto() == 1 && i != y) return false;
		}
		return true;
	}

	/**
	 * Resolve o jogo.
	 */
	public void solve(){
		boolean controle = false; 
		while(getNumCelulasVazias() > 0 && !controle){
			imprime();
			controle = jogada(selecionar(4,4));
			imprime();
			
		}
		if (getNumCelulasVazias() == 0) {
			notifyListeners(new TabuleiroEvent("Ganhou"));
		}
		
	}

	private int getNumCelulasVazias() {
		int vazias = 0;
		for (int i = 0; i < array.length; i++) {
			for (int j = 0; j < array.length; j++) {
				if(array[i][j].getValor()==0) vazias++;
			}
		}
		return vazias;
	}
	
	/**
	 * Seleciona a prxima clula a ser preenchida, com base nas heursticas utilizadas.
	 * 
	 * @param xPadrao
	 * @param yPadrao
	 * @return
	 */
	private Celula selecionar(int xPadrao, int yPadrao) {
		
		// Seleciona as clulas mais restringidas.
		
		LinkedList<Celula> maisRestringidas = getMaisRestringidas(array);
		if (maisRestringidas.isEmpty()) {
			return null;
		}
		
		//Dentre as mais restringidas seleciona-se aquelas que esto em quadrantes menos densos.
		
		LinkedList<Celula> emMenorDensidade = getEmMenorDensidade(maisRestringidas);
		
		//dentre as mais restringidas que esto em quadrantes menos densos, seleciona-se aquelas que esto mais distantes da ltima jogada.
		
		LinkedList<Celula> maisDistantes = getMaisDistantes(emMenorDensidade, xPadrao, yPadrao);
		
		//Caso haja empate, escolhe-se uma clula aleatoriamente dentre as selecionadas.
		
		int index = (int)(Math.random() * maisDistantes.size());
		Celula escolhido = maisDistantes.get(index);
		
		System.out.println("\nescolhido: " + escolhido.getX()+"x"+escolhido.getY() + " => " + escolhido.getValoresPossiveis());
		return escolhido;

	}

	private LinkedList<Celula> getMaisRestringidas(Celula[][] array) {
		int menor = Integer.MAX_VALUE;
		LinkedList<Celula> maisRestringidas = new LinkedList<Celula>();
		
		for (int i = 0; i < array.length; i++) {
			for (int j = 0; j < array.length; j++) {
				if(array[i][j].getCusto() != 0){
					menor = Math.min(menor, array[i][j].getCusto());
				}
			}
		}
		
		for (int i = 0; i < array.length; i++) {
			for (int j = 0; j < array.length; j++) {
				if(array[i][j].getCusto() == menor) {
					maisRestringidas.add(array[i][j]);
				}
			}
		}
		return maisRestringidas;
	}

	private LinkedList<Celula> getEmMenorDensidade(LinkedList<Celula> celulas) {
		int menorDensidade = 9;
		LinkedList<Celula> emMenorDensidade = new LinkedList<Celula>();
		for (Celula celula : celulas) {
			int densidade = calculaDensidadeQuadrante(celula.getX(), celula.getY());
			menorDensidade = Math.min(menorDensidade, densidade);
		}
		for (Celula celula : celulas) {
			int densidade = calculaDensidadeQuadrante(celula.getX(), celula.getY());
			if (densidade == menorDensidade) {
				emMenorDensidade.add(celula);
			}
		}
		return emMenorDensidade;
	}
	
	private int calculaDensidadeQuadrante(int x, int y) {

		int den = 0;
		switch (getQuadrante(x,y)) {
		case 1:

			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++; 
					
				}
			}
			return den;
		case 2:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = tamanho/3 ; j < 2*tamanho/3 ; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 3:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 4:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 5:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 6:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 7:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 8:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		case 9:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 0; j < tamanho/3; j++){
					Celula cel = this.array[i][j];
					if(cel.getValor() != 0) den++;
				}
			}
			return den;
		}
		return den;
	}

	private LinkedList<Celula> getMaisDistantes(LinkedList<Celula> menores, int x, int y) {
		
		double maior = Double.MIN_VALUE;
		LinkedList<Celula> maisDistantes = new LinkedList<Celula>();
		
		for (Celula celula : menores) {
			double distancia = Math.sqrt(Math.pow(x - celula.getX(), 2)+Math.pow(y - celula.getY(), 2));
			maior = Math.max(maior, distancia);
		}
		for (Celula celula : menores) {
			double distancia = Math.sqrt(Math.pow(x - celula.getX(), 2)+Math.pow(y - celula.getY(), 2));
			if(distancia == maior){
				maisDistantes.add(celula);
			}
		}
		return maisDistantes;
		
	}

	private void imprime() {
		for(int i = 0 ; i < tamanho ; i ++){
			System.out.println();
			for(int j = 0 ; j < tamanho ; j++){
				 System.out.print(array[i][j].toString() + "\t");
			}
		}
		System.out.println();
		System.out.println();
	}

	private boolean jogada(Celula celula) {
		
		LinkedList<Integer> valoresAux = celula.getValoresPossiveis();
		boolean controle = false;
		for(int i = 0 ; i < valoresAux.size() ; i++){
			int valor = valoresAux.getFirst().intValue();
			if(setCelula(celula.getX(),celula.getY(),valor)) {
				
				Celula prox = selecionar(this.xAnterior,this.yAnterior);
				boolean result = prox == null || jogada(prox);
				controle = controle || result;
				if(!result) {
					System.out.println(celula.toString() + " valor= " + valor);
					System.out.println("BACKTRACKING!!!\n");
					retornaRestricoes(celula.getX(),celula.getY(),valor);
					valoresAux = celula.getValoresPossiveis();
				}
				else {
					return true;
				}
			}else controle = controle || false;
		}

		return controle;
	}

	private Celula retornaRestricoes(int x, int y, int valor) {
		
		array[x][y].addValor(valor);
		array[x][y].setValor(0);
		
		retornaRestricaoLinha(x,y,valor);
		retornaRestricaoColuna(x,y,valor);
		retornaRestricaoQuadrante(x,y,valor);

		return array[x][y];
		
	}
	private void retornaRestricaoQuadrante(int x, int y, int valor) {
		switch (getQuadrante(x,y)) {
		case 1:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 2:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = tamanho/3 ; j < 2*tamanho/3 ; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor));
				}
			}
			break;
		case 3:
			for(int i = 0 ; i < tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor));
				}
			}
			break;
		case 4:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 5:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 6:
			for(int i = tamanho/3 ; i < 2*tamanho/3 ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 7:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 2*tamanho/3; j < tamanho; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 8:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = tamanho/3; j < 2*tamanho/3; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		case 9:
			for(int i = 2*tamanho/3 ; i < tamanho ; i++){
				for(int j = 0; j < tamanho/3; j++){
					if(i != x && j != y && !array[i][j].isVerificado() && naoRestricoes(i, j, valor, false)) array[i][j].addValor(new Integer(valor)); 
				}
			}
			break;
		}	
	}
	
	private void retornaRestricaoColuna(int x, int y, int valor) {
		for(int i = 0 ; i < tamanho ; i++){
			if(i != x && !array[i][y].isVerificado() && naoRestricoes(i, y, valor, false)){
				array[i][y].addValor(new Integer(valor)); 
			}
		}
		
	}

	private void retornaRestricaoLinha(int x, int y, int valor) {
		for(int i = 0 ; i < tamanho ; i++){
			if(i != y && !array[x][i].isVerificado() && naoRestricoes(x, i, valor, false)){
				array[x][i].addValor(new Integer(valor));
			}
		}
	}

	public boolean addTabuleiroListener(ITabuleiroListener listener) {
		if(this.listeners.contains(listener)) {
			return false;
		}
		return this.listeners.add(listener);
	}
	
	public boolean removeTabuleiroListener(ITabuleiroListener listener) {
		return this.listeners.remove(listener);
	}
	
	public void notifyListeners() {
		TabuleiroEvent event = new TabuleiroEvent(this.array);
		notifyListeners(event);

	}
	
	public void notifyListeners(TabuleiroEvent event) {
		for (ITabuleiroListener listener : this.listeners) {
			listener.update(event);
		}
	}
	
	public Celula getCelula(int x, int y) {
		return array[x][y];
	}

	public int getValor(int x, int y) {
		return this.array[x][y].getValor();
	}

	public Celula[][] getTabuleiro() {
		return this.array;
	}
	
	private void atribuiValor(int x, int y, int valor){
		setCelula(x,y,valor);
		getCelula(x, y).setFixo(true);
	}
	
	private void jogoUm(){
		atribuiValor(0,0,4);
		atribuiValor(0,1,7);
		atribuiValor(1,0,8);
		atribuiValor(0,4,3);
		atribuiValor(0,5,9);
		atribuiValor(0,8,2);
		atribuiValor(1,5,1);
		atribuiValor(2,2,2);
		atribuiValor(2,7,5);
		atribuiValor(3,2,8);
		atribuiValor(3,3,1);
		atribuiValor(3,4,5);
		atribuiValor(3,5,4);
		atribuiValor(3,7,7);
		atribuiValor(4,1,1);
		atribuiValor(4,5,2);
		atribuiValor(4,6,3);
		atribuiValor(5,0,7);
		atribuiValor(5,2,4);
		atribuiValor(5,8,5);
		atribuiValor(6,0,3);
		atribuiValor(6,2,7);
		atribuiValor(6,4,1);
		atribuiValor(6,6,8);
		atribuiValor(7,1,4);
		atribuiValor(7,2,9);
		atribuiValor(7,4,6);
		atribuiValor(8,2,5);
		atribuiValor(8,4,4);
		atribuiValor(8,7,6);
	}
	
	private static void createJogoUm(){
		Tabuleiro jogo = new Tabuleiro(9);
		jogo.setCelula(0,0,4);
		jogo.setCelula(0,1,7);
		jogo.setCelula(1,0,8);
		jogo.setCelula(0,4,3);
		jogo.setCelula(0,5,9);
		jogo.setCelula(0,8,2);
		jogo.setCelula(1,5,1);
		jogo.setCelula(2,2,2);
		jogo.setCelula(2,7,5);
		jogo.setCelula(3,2,8);
		jogo.setCelula(3,3,1);
		jogo.setCelula(3,4,5);
		jogo.setCelula(3,5,4);
		jogo.setCelula(3,7,7);
		jogo.setCelula(4,1,1);
		jogo.setCelula(4,5,2);
		jogo.setCelula(4,6,3);
		jogo.setCelula(5,0,7);
		jogo.setCelula(5,2,4);
		jogo.setCelula(5,8,5);
		jogo.setCelula(6,0,3);
		jogo.setCelula(6,2,7);
		jogo.setCelula(6,4,1);
		jogo.setCelula(6,6,8);
		jogo.setCelula(7,1,4);
		jogo.setCelula(7,2,9);
		jogo.setCelula(7,4,6);
		jogo.setCelula(8,2,5);
		jogo.setCelula(8,4,4);
		jogo.setCelula(8,7,6);
		jogo.solve();
	}

	private void jogoDificil(){
		atribuiValor(0,3,4);
		atribuiValor(1,0,5);
		atribuiValor(1,2,1);
		atribuiValor(1,3,3);
		atribuiValor(1,5,6);
		atribuiValor(2,4,9);
		atribuiValor(2,7,7);
		atribuiValor(2,8,2);
		atribuiValor(3,0,9);
		atribuiValor(3,3,2);
		atribuiValor(3,5,5);
		atribuiValor(3,6,8);
		atribuiValor(3,7,1);
		atribuiValor(4,0,4);
		atribuiValor(4,7,5);
		atribuiValor(4,8,7);
		atribuiValor(5,0,8);
		atribuiValor(6,1,3);
		atribuiValor(6,5,8);
		atribuiValor(6,6,7);
		atribuiValor(7,4,1);
		atribuiValor(7,6,6);
		atribuiValor(8,2,6);
		atribuiValor(8,7,9);
		atribuiValor(8,8,1);
	}
	
	private static void createJogoDificil(){
		Tabuleiro jogo = new Tabuleiro(9);
		jogo.setCelula(0,3,4);
		jogo.setCelula(1,0,5);
		jogo.setCelula(1,2,1);
		jogo.setCelula(1,3,3);
		jogo.setCelula(1,5,6);
		jogo.setCelula(2,4,9);
		jogo.setCelula(2,7,7);
		jogo.setCelula(2,8,2);
		jogo.setCelula(3,0,9);
		jogo.setCelula(3,3,2);
		jogo.setCelula(3,5,5);
		jogo.setCelula(3,6,8);
		jogo.setCelula(3,7,1);
		jogo.setCelula(4,0,4);
		jogo.setCelula(4,7,5);
		jogo.setCelula(4,8,7);
		jogo.setCelula(5,0,8);
		jogo.setCelula(6,1,3);
		jogo.setCelula(6,5,8);
		jogo.setCelula(6,6,7);
		jogo.setCelula(7,4,1);
		jogo.setCelula(7,6,6);
		jogo.setCelula(8,2,6);
		jogo.setCelula(8,7,9);
		jogo.setCelula(8,8,1);
		jogo.solve();
	}
	
	public static void main(String[] args) {
		createJogoUm();
//		createJogoDificil();
	}
	
}
